package ch.frankel.blog.validation;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertSame;

import java.util.Set;

import javax.validation.ConstraintViolation;
import javax.validation.Validation;
import javax.validation.Validator;
import javax.validation.constraints.NotNull;

import org.junit.Before;
import org.junit.Test;

import ch.frankel.blog.validation.model.composition.Person;
import ch.frankel.blog.validation.model.composition.User;
import ch.frankel.blog.validation.validator.Capitalized;
import ch.frankel.blog.validation.validator.CapitalizedNotNullSingle;

public class CompositionValidationTest {

	private Validator validator;

	@Before
	public void setUp() throws Exception {

		validator = Validation.buildDefaultValidatorFactory().getValidator();
	}

	/**
	 * Tests composition validation. Checks there's a unique violation when
	 * <code>firstName</code> is <code>null</code>.
	 */
	@Test
	public void validatePersonNull() {

		Person person = new Person();

		Set<ConstraintViolation<Person>> violations = validator
				.validate(person);

		assertFalse(violations.isEmpty());
		assertEquals(1, violations.size());

		ConstraintViolation<Person> violation = violations.iterator().next();

		assertEquals("firstName", violation.getPropertyPath().iterator().next()
				.getName());
		assertSame(NotNull.class, violation.getConstraintDescriptor()
				.getAnnotation().annotationType());
	}

	/**
	 * Tests composition validation. Checks there's a unique violation when
	 * <code>firstName</code> is not capitalized.
	 */
	@Test
	public void validatePersonNotCapitalized() {

		Person person = new Person();
		
		person.setFirstName("ad");

		Set<ConstraintViolation<Person>> violations = validator
				.validate(person);

		assertFalse(violations.isEmpty());
		assertEquals(1, violations.size());

		ConstraintViolation<Person> violation = violations.iterator().next();

		assertEquals("firstName", violation.getPropertyPath().iterator().next()
				.getName());
		assertSame(Capitalized.class, violation.getConstraintDescriptor()
				.getAnnotation().annotationType());
	}

	/**
	 * Tests composition validation. Checks there's a unique violation when
	 * <code>firstName</code> is <code>null</code>.
	 */
	@Test
	public void validateUserNull() {

		User user = new User();
		
		Set<ConstraintViolation<User>> violations = validator
				.validate(user);

		assertFalse(violations.isEmpty());
		assertEquals(1, violations.size());

		ConstraintViolation<User> violation = violations.iterator().next();

		assertEquals("login", violation.getPropertyPath().iterator().next()
				.getName());
		assertSame(CapitalizedNotNullSingle.class, violation.getConstraintDescriptor()
				.getAnnotation().annotationType());
	}
}
